<?php
error_reporting(E_ALL ^ E_NOTICE);

header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Content-Type: application/json; charset=utf-8");

// Incluir la conexión a PostgreSQL
include_once '../conexion.php';
include_once '../clases/TokenValidator.php';

// Llama la validación antes de cualquier consulta
$usuarioId = TokenValidator::validar($conn);

try {
    // Obtener el cuerpo de la solicitud y decodificar JSON
    $bodyRequest = file_get_contents("php://input");
    $datos = json_decode($bodyRequest, true);

    // Validar que los parámetros sean correctos
    if (!isset($datos[0], $datos[1], $datos[2], $datos[3])) {
        throw new Exception("Parámetros inválidos: Se requieren cuatro valores.");
    }

    $empresa_id      = (int)$datos[0];
    $sucursal_id     = (int)$datos[1];
    $cuenta_id       = (int)$datos[2];
    $fecha           = $datos[3]; // 'YYYY-MM-DD'

    // Saldo inicial = (ingreso - egreso) acumulado antes de :fecha
    // menos la suma de pagos electrónicos (no 'EFECTIVO') asociados a ventas de ese mismo conjunto.
    $sql = "
      WITH dcset AS (  -- movimientos de ESTA caja antes de la fecha
      SELECT dc.movimiento_id, dc.operacion
      FROM detcaja dc
      WHERE dc.empresa_id      = :empresa_id
        AND dc.sucursal_id     = :sucursal_id
        AND dc.cuenta_id       = :cuenta_id
        AND dc.fecha_operacion < :fecha
        AND dc.anulado         = false
    ),
    base AS (
      SELECT COALESCE(SUM(dc.ingreso - dc.egreso), 0) AS saldo
      FROM detcaja dc
      WHERE dc.empresa_id      = :empresa_id
        AND dc.sucursal_id     = :sucursal_id
        AND dc.cuenta_id       = :cuenta_id
        AND dc.fecha_operacion < :fecha
        AND dc.anulado         = false
    ),
    pe AS (  -- pagos electrónicos (NO efectivo) ligados a esos movimientos
      SELECT COALESCE(SUM(dp.monto_pago), 0) AS total_pe
      FROM despacho_pago dp
      LEFT JOIN dcset mv_v
             ON mv_v.operacion = 'VTA' AND dp.venta_id    = mv_v.movimiento_id
      LEFT JOIN dcset mv_c
             ON mv_c.operacion = 'COB' AND dp.cobranza_id = mv_c.movimiento_id
      WHERE dp.activo = false
        AND (mv_v.movimiento_id IS NOT NULL OR mv_c.movimiento_id IS NOT NULL)
        AND UPPER(TRIM(dp.descripcion_pago)) <> 'EFECTIVO'
        AND dp.caja_id = :cuenta_id
        AND DATE(dp.fecha_registro) < :fecha  -- si timestamp
    )
    SELECT (b.saldo - p.total_pe) AS saldo_inicial
    FROM base b CROSS JOIN pe p
    ";

    $stmt = $conn->prepare($sql);
    $stmt->bindValue(':empresa_id', $empresa_id, PDO::PARAM_INT);
    $stmt->bindValue(':sucursal_id', $sucursal_id, PDO::PARAM_INT);
    $stmt->bindValue(':cuenta_id',   $cuenta_id, PDO::PARAM_INT);
    $stmt->bindValue(':fecha',       $fecha, PDO::PARAM_STR);

    // Ejecutar la consulta
    $stmt->execute();
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Imprimir respuesta en formato JSON
    print_json($rows);

} catch (Exception $e) {
    // Manejo de errores
    error_log("❌ Error en Saldo_Inicial: " . $e->getMessage());
    error_response("Error: " . $e->getMessage());
} finally {
    // Cerrar la conexión
    unset($stmt);
    $conn = null;
}

/**
 * Imprime datos en formato JSON y maneja errores de codificación
 */
function print_json($data) {
    echo json_encode($data, JSON_UNESCAPED_UNICODE);

    switch (json_last_error()) {
        case JSON_ERROR_NONE:
            break;
        case JSON_ERROR_DEPTH:
            error_response('Maximum stack depth exceeded');
            break;
        case JSON_ERROR_STATE_MISMATCH:
            error_response('Underflow or the modes mismatch');
            break;
        case JSON_ERROR_CTRL_CHAR:
            error_response('Unexpected control character found');
            break;
        case JSON_ERROR_SYNTAX:
            error_response('Syntax error, malformed JSON');
            break;
        case JSON_ERROR_UTF8:
            error_response('Malformed UTF-8 characters, possibly incorrectly encoded');
            break;
        default:
            error_response('Unknown error');
            break;
    }
}

/**
 * Devuelve un mensaje de error en formato JSON
 */
function error_response($message) {
    http_response_code(500);
    echo json_encode(['error' => $message], JSON_UNESCAPED_UNICODE);
    exit();
}