<?php
error_reporting(E_ALL ^ E_NOTICE);

header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Content-Type: application/json; charset=utf-8");

// Incluir la conexión a PostgreSQL
include_once '../conexion.php';
include_once '../clases/TokenValidator.php';

// Llama la validación antes de cualquier consulta
$usuarioId = TokenValidator::validar($conn);

try {
    // Obtener el cuerpo de la solicitud
    $bodyRequest = file_get_contents("php://input");
    $datos = json_decode($bodyRequest, true);

    // Validar los parámetros antes de ejecutar la consulta
    if (!isset($datos[0]) || !is_numeric($datos[0]) ||
        !isset($datos[1]) || !is_numeric($datos[1]) ||
        !isset($datos[2]) || empty($datos[2])) {
        throw new Exception("Parámetros inválidos.");
    }

    // Convertir valores a tipos adecuados
    $empresaId = (int) $datos[0];
    $sucursalId = (int) $datos[1];
    $nombre = trim($datos[2]);

    // Consulta con parámetros preparados
    $sql = "SELECT nombre 
            FROM clientes 
            WHERE empresa_id = :empresaId 
            AND sucursal_id = :sucursalId 
            AND nombre = :nombre 
            LIMIT 1";

    $stmt = $conn->prepare($sql);
    $stmt->bindValue(':empresaId', $empresaId, PDO::PARAM_INT);
    $stmt->bindValue(':sucursalId', $sucursalId, PDO::PARAM_INT);
    $stmt->bindValue(':nombre', $nombre, PDO::PARAM_STR);
    
    // Ejecutar la consulta
    $stmt->execute();
    
    // Obtener los resultados
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Imprimir respuesta en formato JSON
    print_json($rows);

} catch (Exception $e) {
    // Manejo de errores
    error_response("Error: " . $e->getMessage());
} finally {
    // Cerrar la conexión
    $conn = null;
}

/**
 * Imprime datos en formato JSON y maneja errores de codificación
 */
function print_json($data) {
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    
    switch (json_last_error()) {
        case JSON_ERROR_NONE:
            break;
        case JSON_ERROR_DEPTH:
            error_response('Maximum stack depth exceeded');
            break;
        case JSON_ERROR_STATE_MISMATCH:
            error_response('Underflow or the modes mismatch');
            break;
        case JSON_ERROR_CTRL_CHAR:
            error_response('Unexpected control character found');
            break;
        case JSON_ERROR_SYNTAX:
            error_response('Syntax error, malformed JSON');
            break;
        case JSON_ERROR_UTF8:
            error_response('Malformed UTF-8 characters, possibly incorrectly encoded');
            break;
        default:
            error_response('Unknown error');
            break;
    }
}

/**
 * Devuelve un mensaje de error en formato JSON
 */
function error_response($message) {
    http_response_code(500);
    echo json_encode(['error' => $message], JSON_UNESCAPED_UNICODE);
    exit();
}
?>
