<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header("Content-Type: application/json; charset=utf-8");

include_once '../conexion.php';
include_once '../clases/TokenValidator.php';

// ✅ Validación de token
$usuarioId = TokenValidator::validar($conn);

try {
    $datos = json_decode(file_get_contents("php://input"), true);

    if (!isset($datos[0], $datos[1], $datos[2])) {
        throw new Exception("Parámetros incompletos.");
    }

    $usuario_id = (int)$datos[0];
    $empresa_id = (int)$datos[1];
    $sucursal_id = (int)$datos[2];

    // 🔍 1. Buscar turno ABIERTO actual (sin importar quién lo abrió)
    $sqlTurnoAbierto = "
        SELECT ta.*, t.nombre AS nombre_turno
        FROM turno_apertura ta
        JOIN turno t ON t.id = ta.turno_id
        WHERE ta.estado = 'ABIERTO'
        AND ta.empresa_id = :empresa_id
        AND ta.sucursal_id = :sucursal_id
        LIMIT 1
    ";
    $stmtTurnoAbierto = $conn->prepare($sqlTurnoAbierto);
    $stmtTurnoAbierto->execute([
        ':empresa_id' => $empresa_id,
        ':sucursal_id' => $sucursal_id
    ]);
    $turnoAbierto = $stmtTurnoAbierto->fetch(PDO::FETCH_ASSOC);

    if ($turnoAbierto) {
        $turnoIdAbierto = $turnoAbierto['id'];
        $pertenece = ($turnoAbierto['usuario_apertura_id'] == $usuario_id) || existe_usuario_en_turno_detalle($conn, $turnoIdAbierto, $usuario_id);

        if ($pertenece) {
            echo json_encode([
                "ESTADO" => "1",
                "ID_TURNO" => $turnoAbierto['id'],
                "NOMBRE_TURNO" => $turnoAbierto['nombre_turno'],
                "PERTENECE_USUARIO" => true,
                "data" => [$turnoAbierto]
            ]);
            exit;
        } else {
            echo json_encode([
                "ESTADO" => "2",
                "MENSAJE" => "Ya existe un turno aperturado por otro usuario.",
                "ID_TURNO" => 0,
                "data" => []
            ]);
            exit;
        }
    }

    // 🔍 2. Verificar si el usuario tiene un turno ABIERTO o GENERADO
    $sql = "SELECT * FROM verifica_turno_abierto(:usuario_id, :empresa_id, :sucursal_id)";
    $stmt = $conn->prepare($sql);
    $stmt->execute([
        ':usuario_id' => $usuario_id,
        ':empresa_id' => $empresa_id,
        ':sucursal_id' => $sucursal_id
    ]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($row && isset($row['id'])) {
        // ✅ Verificar si es el usuario principal o está asignado en el detalle del turno
        $pertenece = ($row['usuario_id'] == $usuario_id) || existe_usuario_en_turno_detalle($conn, $row['id'], $usuario_id);

        echo json_encode([
            "ESTADO" => "1",
            "ID_TURNO" => $row['id'],
            "NOMBRE_TURNO" => $row['nombre_turno'],
            "PERTENECE_USUARIO" => $pertenece,
            "data" => [$row]
        ]);
        exit;
    }

    // 🔍 3. Buscar turno más antiguo en estado GENERADO
    $sqlAntiguo = "
        SELECT ta.*, t.nombre AS nombre_turno
        FROM turno_apertura ta
        JOIN turno t ON t.id = ta.turno_id
        WHERE ta.estado = 'GENERADO'
        AND ta.empresa_id = :empresa_id
        AND ta.sucursal_id = :sucursal_id
        ORDER BY ta.fecha_inicio ASC, ta.turno_id ASC
        LIMIT 1
    ";
    $stmtAntiguo = $conn->prepare($sqlAntiguo);
    $stmtAntiguo->execute([
        ':empresa_id' => $empresa_id,
        ':sucursal_id' => $sucursal_id
    ]);
    $turnoAntiguo = $stmtAntiguo->fetch(PDO::FETCH_ASSOC);

    if ($turnoAntiguo) {
        $turnoIdAntiguo = $turnoAntiguo['id'];

        // ✅ Verificar si está autorizado en el detalle del turno
        $tienePermiso = existe_usuario_en_turno_detalle($conn, $turnoIdAntiguo, $usuario_id);

        if ($tienePermiso) {
            echo json_encode([
                "ESTADO" => "0",
                "ID_TURNO" => $turnoAntiguo['id'],
                "NOMBRE_TURNO" => $turnoAntiguo['nombre_turno'],
                "PERTENECE_USUARIO" => true,
                "data" => [$turnoAntiguo]
            ]);
        } else {
            echo json_encode([
                "ESTADO" => "3",
                "ID_TURNO" => 0,
                "MENSAJE" => "El turno generado más antiguo no le pertenece al usuario.",
                "data" => []
            ]);
        }
        exit;
    }

    // 🚫 No hay turnos abiertos ni generados
    echo json_encode([
        "ESTADO" => "0",
        "ID_TURNO" => 0,
        "MENSAJE" => "No hay turnos abiertos ni generados.",
        "data" => []
    ]);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'ESTADO' => 'ERROR',
        'MENSAJE' => $e->getMessage()
    ]);
} finally {
    unset($stmt);
    $conn = null;
}

// ✅ Función auxiliar para verificar si el usuario está asignado en el detalle del turno
function existe_usuario_en_turno_detalle($conn, $turno_id, $usuario_id) {
    $sql = "SELECT 1 FROM turno_detalle WHERE turno_apertura_id = :turno_id AND usuario_id = :usuario_id LIMIT 1";
    $stmt = $conn->prepare($sql);
    $stmt->execute([
        ':turno_id' => $turno_id,
        ':usuario_id' => $usuario_id
    ]);
    return $stmt->fetchColumn() ? true : false;
}