<?php
error_reporting(E_ALL ^ E_NOTICE);

header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Content-Type: application/json; charset=utf-8");

include_once '../conexion.php';
include_once '../clases/TokenValidator.php';

// Valida token antes de cualquier operación
$usuarioId = TokenValidator::validar($conn);

try {
    // Leer cuerpo de la solicitud
    $bodyRequest = file_get_contents("php://input");
    $datos = json_decode($bodyRequest, true);

    // Esperamos: [empresa_id, sucursal_id, cliente_id, es_cancelado]
    if (!isset($datos[0], $datos[1], $datos[2], $datos[3]) || 
        !is_numeric($datos[0]) || 
        !is_numeric($datos[1]) || 
        !is_numeric($datos[2]) || 
        !is_bool($datos[3])) {
        throw new Exception("Parámetros inválidos.");
    }

    $param1 = (int) $datos[0];      // empresa_id
    $param2 = (int) $datos[1];      // sucursal_id
    $param3 = (int) $datos[2];      // cliente_id (0 = todos)
    $param4 = (bool) $datos[3];     // es_cancelado (true = incluye cancelados, false = solo vigentes)

    // Llamar a la función PostgreSQL que solo devuelve vencidos
    $sql = "SELECT * 
            FROM listar_creditosxcob_vencidos(:param1, :param2, :param3, :param4)";

    $stmt = $conn->prepare($sql);
    $stmt->bindValue(':param1', $param1, PDO::PARAM_INT);
    $stmt->bindValue(':param2', $param2, PDO::PARAM_INT);
    $stmt->bindValue(':param3', $param3, PDO::PARAM_INT);
    $stmt->bindValue(':param4', $param4, PDO::PARAM_BOOL);

    $stmt->execute();

    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    print_json($rows);

} catch (Exception $e) {
    error_response("Error: " . $e->getMessage());
} finally {
    $conn = null;
}

/**
 * Imprime datos en formato JSON y maneja errores de codificación
 */
function print_json($data) {
    echo json_encode($data, JSON_UNESCAPED_UNICODE);

    switch (json_last_error()) {
        case JSON_ERROR_NONE:
            break;
        case JSON_ERROR_DEPTH:
            error_response('Maximum stack depth exceeded');
            break;
        case JSON_ERROR_STATE_MISMATCH:
            error_response('Underflow or the modes mismatch');
            break;
        case JSON_ERROR_CTRL_CHAR:
            error_response('Unexpected control character found');
            break;
        case JSON_ERROR_SYNTAX:
            error_response('Syntax error, malformed JSON');
            break;
        case JSON_ERROR_UTF8:
            error_response('Malformed UTF-8 characters, possibly incorrectly encoded');
            break;
        default:
            error_response('Unknown error');
            break;
    }
}

/**
 * Devuelve un mensaje de error en formato JSON
 */
function error_response($message) {
    http_response_code(500);
    echo json_encode(['error' => $message], JSON_UNESCAPED_UNICODE);
    exit();
}