<?php
error_reporting(E_ALL ^ E_NOTICE);

header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Content-Type: application/json; charset=utf-8");

include_once '../conexion.php';
include_once '../clases/TokenValidator.php';

// Valida token antes de cualquier operación
$usuarioId = TokenValidator::validar($conn);

try {
    // Leer cuerpo JSON
    $bodyRequest = file_get_contents("php://input");
    $datos = json_decode($bodyRequest, true);

    if (!is_array($datos) || !isset($datos[0]) || !is_string($datos[0])) {
        throw new Exception("Parámetro de filtro inválido.");
    }

    // Esta cadena viene armada desde Java (cFiltro)
    // Ejemplo: "(C.anulado = 0) AND (C.empresa_id = 1) AND (C.cliente_id = 123) AND (((deuda_soles+interes_soles)-pago_soles) > 0)"
    $filtro = $datos[0];

    // IMPORTANTE: estamos confiando en que el filtro lo genera tu sistema,
    // igual que antes. No viene del usuario final.
    $sql = "
        SELECT
            C.fecha,
            C.nserie || '-' || C.ndocumento AS ndocumento,
            C.razon_social,
            C.direccion,
            C.fecha_vencimiento,
            C.deuda_soles,
            C.interes_soles,
            C.pago_soles,
            C.deuda_dolar,
            C.interes_dolar,
            C.pago_dolar,
            CASE
                WHEN C.moneda = '01'
                    THEN (C.deuda_soles + C.interes_soles) - C.pago_soles
                ELSE 0
            END AS saldo_soles,
            CASE
                WHEN C.moneda = '01'
                    THEN 0
                ELSE (C.deuda_dolar + C.interes_dolar) - C.pago_dolar
            END AS saldo_dolar,
            -- igual que TIMESTAMPDIFF(DAY, NOW(), fecha_vencimiento) en MySQL:
            (C.fecha_vencimiento - CURRENT_DATE) AS nro_dias,
            C.sucursal_id
        FROM creditos_cobrar AS C
        INNER JOIN zonas AS Z ON C.zona_id = Z.id
        WHERE ($filtro)
          -- solo vencidos: fecha_vencimiento <= hoy
          AND (CURRENT_DATE - C.fecha_vencimiento) >= 0
        ORDER BY C.fecha ASC, C.nserie, C.ndocumento
    ";

    // Ejecutar consulta con PDO
    $stmt = $conn->query($sql);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    print_json($rows);

} catch (Exception $e) {
    error_response("Error: " . $e->getMessage());
} finally {
    // Cerrar conexión
    $conn = null;
}

/**
 * Imprime datos en formato JSON y maneja errores de codificación
 */
function print_json($data) {
    echo json_encode($data, JSON_UNESCAPED_UNICODE);

    switch (json_last_error()) {
        case JSON_ERROR_NONE:
            break;
        case JSON_ERROR_DEPTH:
            error_response('Maximum stack depth exceeded');
            break;
        case JSON_ERROR_STATE_MISMATCH:
            error_response('Underflow or the modes mismatch');
            break;
        case JSON_ERROR_CTRL_CHAR:
            error_response('Unexpected control character found');
            break;
        case JSON_ERROR_SYNTAX:
            error_response('Syntax error, malformed JSON');
            break;
        case JSON_ERROR_UTF8:
            error_response('Malformed UTF-8 characters, possibly incorrectly encoded');
            break;
        default:
            error_response('Unknown error');
            break;
    }
}

/**
 * Devuelve un mensaje de error en formato JSON
 */
function error_response($message) {
    http_response_code(500);
    echo json_encode(['error' => $message], JSON_UNESCAPED_UNICODE);
    exit();
}