<?php
error_reporting(E_ALL ^ E_NOTICE);

header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Content-Type: application/json; charset=utf-8");

// Se incluye el archivo de conexión
include_once '../conexion.php';
include_once '../clases/TokenValidator.php';

// Llama la validación antes de cualquier consulta
$usuarioId = TokenValidator::validar($conn);

$bodyRequest = file_get_contents("php://input");
$datos = json_decode($bodyRequest, true);

$rows = [];

try {
    // Consulta adaptada a PostgreSQL con alias para evitar ambigüedades
    $sql = "SELECT v.periodo, v.mes, 
                   TO_CHAR(v.fecha_emision, 'Mon') AS nommesabrev, 
                   SUM(CASE WHEN v.docfiscal_id <> 5 THEN v.total ELSE v.total * -1 END) AS total_ventas, 
                   TO_CHAR(v.fecha_emision, 'Month') AS nommes, 
                   v.empresa_id
            FROM ventas v
            WHERE v.periodo = :periodo 
              AND v.empresa_id = :empresa_id
              AND v.sucursal_id = :sucursal_id
              AND v.anulado = 0 
              AND v.afecta_kardex = 1
            GROUP BY v.empresa_id, v.periodo, v.mes, v.fecha_emision";

    $stmt = $conn->prepare($sql);
    $stmt->bindParam(':periodo', $datos[0], PDO::PARAM_STR);
    $stmt->bindParam(':empresa_id', $datos[1], PDO::PARAM_INT);
    $stmt->bindParam(':sucursal_id', $datos[2], PDO::PARAM_INT);

    // Ejecutar la consulta
    $stmt->execute();
    
    // Obtener resultados
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    error_response("Error en la consulta: " . $e->getMessage());
}

// Cerrar la conexión
$conn = null;

// Imprimir la respuesta en formato JSON
print_json($rows);

function print_json($data) {
    echo json_encode($data, JSON_UNESCAPED_UNICODE);

    switch (json_last_error()) {
        case JSON_ERROR_NONE:
            break;
        case JSON_ERROR_DEPTH:
            error_response('Maximum stack depth exceeded');
            break;
        case JSON_ERROR_STATE_MISMATCH:
            error_response('Underflow or the modes mismatch');
            break;
        case JSON_ERROR_CTRL_CHAR:
            error_response('Unexpected control character found');
            break;
        case JSON_ERROR_SYNTAX:
            error_response('Syntax error, malformed JSON');
            break;
        case JSON_ERROR_UTF8:
            error_response('Malformed UTF-8 characters, possibly incorrectly encoded');
            break;
        default:
            error_response('Unknown error');
            break;
    }
}

function error_response($message) {
    http_response_code(500);
    echo json_encode(['error' => $message], JSON_UNESCAPED_UNICODE);
    exit();
}
?>
