<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include_once __DIR__ . '/../../conexion.php';
require_once __DIR__ . '/../../tcpdf/tcpdf.php';

header("Content-Type: application/pdf; charset=utf-8");

try {
    $bodyRequest = file_get_contents("php://input");
    $datos = json_decode($bodyRequest, true);

    if (!isset($datos[0], $datos[1], $datos[2], $datos[3])) {
        throw new Exception("Falta la fecha_turno.");
    }

    $fechaInicio = $datos[0];
    $fechaFin    = $datos[1];
    $empresaId   = (int) $datos[2];
    $sucursalId  = (int) $datos[3];

    // Consulta con JOIN y agrupación por cliente_id
    $sql = "SELECT 
                v.cliente_id,
                substring(unaccent('unaccent',c.nombre),0,65) as nombre,
                SUM(d.galones) AS total_galones,
                SUM(d.soles) AS total_credito
            FROM controlador.\"DESPACHOS\" d
            INNER JOIN ventas v ON v.despacho_id = d.id
            INNER JOIN clientes c ON c.id = v.cliente_id
            WHERE d.operacion = 'CRD'
            AND d.fecha_turno BETWEEN :fecha_inicio AND :fecha_fin
            AND d.empresa_id = :empresa_id
            AND d.sucursal_id = :sucursal_id
            GROUP BY v.cliente_id, c.nombre
            ORDER BY c.nombre";

    $stmt = $conn->prepare($sql);
    $stmt->execute([
        ':fecha_inicio' => $fechaInicio,
        ':fecha_fin'    => $fechaFin,
        ':empresa_id'   => $empresaId,
        ':sucursal_id'  => $sucursalId
    ]);

    $creditos = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($creditos)) {
        throw new Exception("No se encontraron créditos entregados con ventas.");
    }

    // Crear PDF
    $pdf = new TCPDF('P', 'mm', 'A4', true, 'UTF-8', false);
    $pdf->SetMargins(10, 10, 10);
    $pdf->SetAutoPageBreak(true, 10);
    $pdf->SetDisplayMode(100, 'SinglePage');
    $pdf->AddPage();

    $logo_path = '../../tcpdf/logos/10278514876.png';
    if (file_exists($logo_path)) {
        $pdf->Image($logo_path, '', '', 30, 15, '', '', 'T', false, 300, 'C', false, false, 0, false, false, false);
    }
    $pdf->Ln(18);

    // --- Cabecera ---
    $pdf->SetFont('helvetica', 'B', 14);
    $pdf->Cell(0, 10, 'REPORTE DE CREDITOS ENTREGADOS (AGRUPADO POR CLIENTE)', 0, 1, 'C');
    $pdf->Ln(5);

    // --- Tabla ---
    $pdf->SetFillColor(5, 152, 172);       // Fondo celeste oscuro
    $pdf->SetTextColor(255, 255, 255);     // Letras blancas
    $pdf->SetDrawColor(0, 0, 0);           // Borde negro
    $pdf->SetFont('helvetica', 'B', 10);

    $pdf->Cell(15, 8, 'ID', 1, 0, 'C', true);
    $pdf->Cell(120, 8, 'Cliente', 1, 0, 'C', true);
    $pdf->Cell(30, 8, 'Galones', 1, 0, 'C', true);
    $pdf->Cell(30, 8, 'Crédito (S/)', 1, 1, 'C', true);

    $pdf->SetTextColor(0, 0, 0);

    $pdf->SetFont('helvetica', '', 8);

    $totalGalones = 0;
    $totalGeneral = 0;

    foreach ($creditos as $row) {
        $pdf->Cell(15, 7, $row['cliente_id'], 1, 0, 'C');
        $pdf->Cell(120, 7, utf8_decode($row['nombre']), 1, 0, 'L');
        $pdf->Cell(30, 7, number_format($row['total_galones'], 3), 1, 0, 'R');
        $pdf->Cell(30, 7, number_format($row['total_credito'], 2), 1, 1, 'R');

        $totalGalones += $row['total_galones'];
        $totalGeneral += $row['total_credito'];
    }
    // Línea vacía antes del total
    $pdf->Ln(2);

    $pdf->SetFont('helvetica', 'B', 10);
    $pdf->Cell(135, 8, 'TOTAL GENERAL', 1, 0, 'R');
    $pdf->Cell(30, 8, number_format($totalGalones, 3), 1, 0, 'R');
    $pdf->Cell(30, 8, number_format($totalGeneral, 2), 1, 1, 'R');

    $pdf->Output('reporte_creditos_cliente.pdf', 'I');

} catch (Exception $e) {
    http_response_code(500); // IMPORTANTE
    header("Content-Type: application/json; charset=utf-8"); // CORRIGE tipo de respuesta
    
    error_log("ERROR: " . $e->getMessage());
    echo json_encode([
        "estado" => "0",
        "mensaje" => $e->getMessage()
    ]);
}
?>
