<?php
error_reporting(E_ALL ^ E_NOTICE);

// Permite la ejecución de los métodos
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Content-Type: application/json; charset=utf-8");

// Si quieres permitir CORS como en la primera API, descomenta:
// header("Access-Control-Allow-Origin: *");

// Conexión a PostgreSQL (PDO)
include_once '../conexion.php';
include_once '../clases/TokenValidator.php';

// Valida el token ANTES de procesar
$usuarioId = TokenValidator::validar($conn);

// Lee el cuerpo de la solicitud
$bodyRequest = file_get_contents("php://input");
$datos = json_decode($bodyRequest, true);

$rows = array();

try {
    // Validación básica de parámetros
    if (!is_array($datos) || count($datos) < 4) {
        throw new Exception("Parámetros insuficientes. Se esperan [empresa_id, sucursal_id, fecha_ini, fecha_fin]");
    }

    $empresaId  = (int)$datos[0];
    $sucursalId = (int)$datos[1];
    $fechaIni   = $datos[2];
    $fechaFin   = $datos[3];

    $sql = "SELECT public.listar_margen(:empresa_id, :sucursal_id, :fecha_ini, :fecha_fin) AS margen_total;";

    $stmt = $conn->prepare($sql);

    // Vinculamos parámetros
    $stmt->bindValue(':empresa_id',  $empresaId,  PDO::PARAM_INT);
    $stmt->bindValue(':sucursal_id', $sucursalId, PDO::PARAM_INT);
    $stmt->bindValue(':fecha_ini',   $fechaIni,   PDO::PARAM_STR);
    $stmt->bindValue(':fecha_fin',   $fechaFin,   PDO::PARAM_STR);

    // Ejecutamos
    $stmt->execute();

    // Obtenemos el resultado (una sola fila con margen_total)
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Si no hay filas, devolvemos un array vacío (por compatibilidad)
    if (empty($rows)) {
        $rows = [];
    }
} catch (PDOException $e) {
    error_response("Error al ejecutar la consulta: " . $e->getMessage());
} catch (Exception $e) {
    error_response($e->getMessage());
}

// Cerramos conexión
$conn = null;

// Imprimimos JSON
print_json($rows);

function print_json($data) {
    echo json_encode($data, JSON_UNESCAPED_UNICODE);

    switch (json_last_error()) {
        case JSON_ERROR_NONE:
            break;
        case JSON_ERROR_DEPTH:
            error_response('Maximum stack depth exceeded');
            break;
        case JSON_ERROR_STATE_MISMATCH:
            error_response('Underflow or the modes mismatch');
            break;
        case JSON_ERROR_CTRL_CHAR:
            error_response('Unexpected control character found');
            break;
        case JSON_ERROR_SYNTAX:
            error_response('Syntax error, malformed JSON');
            break;
        case JSON_ERROR_UTF8:
            error_response('Malformed UTF-8 characters, possibly incorrectly encoded');
            break;
        default:
            error_response('Unknown JSON error');
            break;
    }
}

function error_response($message) {
    http_response_code(500);
    echo json_encode(['error' => $message], JSON_UNESCAPED_UNICODE);
    exit();
}
?>